//
// taskswapper - Maemo5 hotkey based task swapper
//
// Copyright (C) 2010 Sami Väisänen   samiv@ensisoft.com
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  
//
//
//
// $Id: main_x11.cpp,v 1.7 2010/02/18 19:21:26 svaisane Exp $
//
// building
// g++ -Wall -g -O0 -lX11 -o foobar main.cpp (for debug)
// g++ -Wall -03 -lX11 -o taskswapper main_x11.cpp (for release)
//
// useful tools:
//
//  wmtrcl  --UNIX/Linux command line tool to interact with an EWMH/NetWM compatible X Window Manager. 
//              to activate a window:           wmctrl -i -a 0x235232
//              to list top level windows:      wmctrl -l 
//
//  xev     --print x events
//
//  xprop   --print x server properties 
//              to list root properties:          xprop -root
//              to get list of top level windows: xprop -root | grep _NET_CLIENT_LIST_STACKING
//
//  xwininfo  --print window information 
//              window property tree starting at root: xwininfo -root -tree
//
//

#include <set>
#include <iostream>
#include <cassert>
#include <cstdlib>
#include <cerrno>
#include <cstring>
#include <vector>
#include <fstream>
#include <algorithm>
#include <stdexcept>
#include <sys/select.h>
#include <X11/Xlib.h>
#include <X11/Xatom.h>
#include <unistd.h>
#include "version.h"

using namespace std;

namespace {

struct hotkey {
    unsigned modifier;
    unsigned key;
};

struct app_context {
    Display* dpy;
    hotkey next_window;
    hotkey prev_window;
    set<Window> visible;
    int window_index;
};

// allowed types for T are fundamental types, 8, 16 and 32 bits
// todo: could use some simple metaprogramming to enforce this
// (dont want to use boost in order to keep the dependecies minimal)
template<typename T>
vector<T> get_property(Display* dpy, Window wnd, const char* propname, Atom property_type)
{
    Atom xprop_name = XInternAtom(dpy, propname, False);
    Atom xprop_return_type = 0;
    int return_format             = 0;        // return format (in bit size i belive)
    unsigned long nitems_return   = 0;        // the actual number of 8, 16 or 32 bit items stored in the return data
    unsigned long bytes_remaining = 0;        // the number of bytes remaining after doing a partial read
    unsigned char* property_data  = NULL;     // the actual data returned by this monster function
    
    const int MAX_PROPERTY_VALUE_LEN = 1024; // this is 4kb in 32 bit quantities
    
    // call this do-it-all mosnter of a function and save the return
    // values into a vector on success
    if (XGetWindowProperty(
            dpy,
            wnd,
            xprop_name, 
            0,                          // specified property offset in 32bit quantities
            MAX_PROPERTY_VALUE_LEN,     // 32bit units!
            False,                      // don't delete the property kthx
            property_type,      
            &xprop_return_type,
            &return_format,
            &nitems_return,
            &bytes_remaining,
            &property_data) != Success)
        throw std::runtime_error("failed to get property:");

    if (!property_data)
        return vector<T>();

    assert(xprop_return_type == property_type);
    assert(return_format == 8  || return_format == 16 || return_format == 32);
    assert(sizeof(T) == return_format / 8);

    vector<T> ret;
    ret.resize(nitems_return);
    memcpy(&ret[0], property_data, nitems_return * return_format / 8);
    
    XFree(property_data);

    return ret;
}

void send_activate_event(Display* disp, Window win)
{
    XEvent event = {};
    event.xclient.type         = ClientMessage;
    event.xclient.serial       = 0;
    event.xclient.send_event   = True;
    event.xclient.message_type = XInternAtom(disp, "_NET_ACTIVE_WINDOW", False);
    event.xclient.window       = win;
    event.xclient.format       = 32; // sending a list of longs (32bit values in the message). has to be set
                                     // actual data is all zeroes, and is not used.
    
    XSendEvent(disp, DefaultRootWindow(disp), False, SubstructureRedirectMask | SubstructureNotifyMask, &event);

    XMapRaised(disp, win);
}

vector<Window> get_toplevel_windows(Display* dpy)
{
    // get a list of toplevel windows. Unfortunately on Maemo
    // this list also includes a whole bunch of windows that do not have
    // a window visible to the user
    vector<Window> top_level = get_property<Window>(dpy, XDefaultRootWindow(dpy), "_NET_CLIENT_LIST", XA_WINDOW);
    
    return top_level;
}


// Generates X error BadWindow which is silently discarded
// in our custom error handler
  /*
Window find_toplevel_visible_child(Display* dpy, Window wnd)
{
    Window root;
    Window parent;
    Window* children = NULL;
    unsigned int children_count = 0;
    Window ret = 0;

    XQueryTree(dpy, wnd, &root, &parent, &children, &children_count);
    if (children)
    {
        for (unsigned int i=0; i<children_count; ++i)
        {
            Window child = children[i];
            ret = find_toplevel_visible_child(dpy, child);
            if (ret != 0)
                break;
        }
        XFree(children);
    }
    else
    {
        vector<Window> top = get_toplevel_windows(dpy);
        if (std::find(top.begin(), top.end(), wnd) != top.end())
            ret = wnd;
    }
    return ret;
}
  */


// Generates X error  BadWindow which is silently discarded
// in our custom error handler
Window find_toplevel_visible_child(Display* dpy, Window wnd)
{
    Window root      = 0;
    Window parent    = 0;
    Window* children = NULL;
    unsigned int children_count = 0;
    Window ret = 0;

    XQueryTree(dpy, wnd, &root, &parent, &children, &children_count);
    if (children)
    {
        vector<Window> top  = get_toplevel_windows(dpy);
        for (unsigned int i=0; i<children_count; ++i)
        {
            if (std::find(top.begin(), top.end(), children[i]) != top.end())
            {
                ret = children[i];
                break;
            }
        }
        XFree(children);
    }
    return ret;
}

vector<Window> get_visible_windows(app_context& ctx)
{
    // get all toplevel windows, then guess which ones are visible
    vector<Window> toplevel = get_toplevel_windows(ctx.dpy);

    vector<Window> visible;
    // check the returned windows if they have  hildon specic properties set
    // indicating that they are stackable or they have a menu. 
    // If either property is set we assume that this window is a top level window.
    for (vector<Window>::iterator it = toplevel.begin(); it != toplevel.end(); ++it)
    {
        // first check for the menu prop, most top level windows have a menu I presume
        vector<int> x = get_property<int>(ctx.dpy, *it, "_HILDON_WM_WINDOW_MENU_INDICATOR", XA_INTEGER);
        if (!x.empty() && x[0])
        {
            visible.push_back(*it);
            continue;
        }

        x = get_property<int>(ctx.dpy, *it, "_HILDON_STACKABLE_WINDOW", XA_INTEGER);
        if (!x.empty())
        {
            visible.push_back(*it);
            continue;
        }
        if (ctx.visible.find(*it) != ctx.visible.end())
            visible.push_back(*it);
    }
    
    // finally sort the windows so that the order (sequence)
    // of windows is predictable while not necessarily WYSIWYG
    sort(visible.begin(), visible.end());
    return visible;
}
  
void focus_window(app_context& ctx)
{
    vector<Window> visible = get_visible_windows(ctx);
    if (visible.empty())
        return;
    
    if (ctx.window_index < 0)
        ctx.window_index = visible.size() - 1;
    if (ctx.window_index >= (int)visible.size())
        ctx.window_index = 0;

    send_activate_event(ctx.dpy, visible.at(ctx.window_index));
}


void process_events(app_context& ctx)
{
    while (XPending(ctx.dpy))
    {
        XEvent ev;
        XNextEvent(ctx.dpy, &ev);
        switch (ev.type)
        {
            case KeyPress:
                //cout << "KeyPress event!" << endl;
                if (ev.xkey.keycode == ctx.next_window.key && 
                    ev.xkey.state   == ctx.next_window.modifier)
                    ++ctx.window_index;

                if (ev.xkey.keycode == ctx.prev_window.key && 
                    ev.xkey.state   == ctx.prev_window.modifier)
                    --ctx.window_index;
                
                focus_window(ctx);
                break;

            // The idea here is to track window creations/mappings in order to catch new top level window ids so that 
            // we can swap into them. If the windows are created without menus and they are not stackable windows
            // they dont have the _HILDON_STACKABLE_WINDOW and _HILDON_WM_WINDOW_MENU_INDICATOR properties set
            // which means we can't identify them.
            // todo: if this works reliably there's no need to check the window lists repeatedly
            case MapNotify:
                //cout << "MapNotify event" << endl;
                {
                    Window visible_child = find_toplevel_visible_child(ctx.dpy, ev.xmap.window);
                    if (visible_child)
                    {
                        ctx.visible.insert(visible_child);
                        //cout << "inserted new window: " << visible_child << endl;
                    }
                }
                break;
                
            case CreateNotify:
                //cout << "CreateNotify event" << endl;
                break;

            case DestroyNotify:
                //cout << "DestroyNotify event!" << endl;
                ctx.visible.erase(ev.xdestroywindow.window);
                break;
                
          // focus events are not being sent on Maemo
          //case FocusIn:
             //cout << "FocusIn event" << endl;
             //break;
          //case FocusOut:
             //cout << "FocusOut event" << endl;
             //break;
        }
    }
}

hotkey parse_key(Display* dpy, const string& str)
{
    string::size_type pos = str.find(":");
    if (pos == string::npos)
        return hotkey();

    string modifiers = str.substr(0, pos++);
    string symkey    = str.substr(pos);
    
    int mods = 0;
    if (modifiers.find("Ctrl") != string::npos)
        mods |= ControlMask;
    if (modifiers.find("Shift") != string::npos)
        mods |= ShiftMask;
    if (modifiers.find("Alt") != string::npos)
        mods |= Mod1Mask;

    hotkey key;
    key.modifier = mods;
    key.key      = XKeysymToKeycode(dpy, XStringToKeysym(symkey.c_str()));
    return key;
}
void print_help()
{
    cout << "\ntaskswapper - Maemo5 hotkey based task swapper\n";
    cout << TASK_SWAPPER_VERSION "  " __DATE__ "\n";
    cout << "Copyright (c) 2010 Sami Väisänen\n";
    cout << "samiv@ensisoft.com\n\n";
    cout << "--help     -- print this help\n";
    cout << "--next     -- setup \"next window\" hotkey\n";
    cout << "--prev     -- setup \"previous window\" hotkey\n\n";
    cout << "hotkeys should be in \"modifier+modifier:key\" format\n";
    cout << "for example:\n";
    cout << "--next Shift:Left\n";
    cout << "--prev Shift+Ctrl:Left\n\n";
    cout << "Available modifiers: Ctrl, Alt, Shift\n\n";
}

bool find_argument(int argc, char* argv[], const char* name, string& value)
{
    for (int i=0; i<argc; ++i)
    {
        if (!strcmp(argv[i], name))
        {
            if (i+1 <  argc)
                value = argv[i+1];
            return true;
        }
    }
    return false;
}

  // custom X11 error handler. Return value is ignored
int x_error_handler(Display* dpy, XErrorEvent* err)
{
    // XQueryTree generates BadWindow requests
    // there's probably a good reason for it but I can't 
    // figure out what. Therefore as a very ugly workaround
    // such errors are just silently discarded
    if (err->error_code == BadWindow)
        return 0; 
        
    // oops something else went wrong, fall back on default handler.
    // First call with NULL replaces this handler with default handler
    // and returns the current handler, i.e this function
    int (*handler)(Display*, XErrorEvent*) = XSetErrorHandler(NULL);
    
    assert(handler == x_error_handler);
    
    handler = XSetErrorHandler(NULL);
    
    assert(handler != x_error_handler);
    
    // call default handler
    return handler(dpy, err);
    
}

} // namespace

int main(int argc, char* argv[])
{
    //freopen("/var/log/swapper.log", "w", stdout);
    //freopen("/var/log/swapper.log", "w", stderr);

    string s;
    if (find_argument(argc, argv, "--help", s))
    {
        print_help();
        return EXIT_SUCCESS;
    };

    app_context ctx;
    ctx.window_index = 0;
    ctx.dpy = NULL;
    memset(&ctx.next_window, 0, sizeof(hotkey));
    memset(&ctx.prev_window, 0, sizeof(hotkey));

    // get connection to the server
    ctx.dpy = XOpenDisplay(NULL);
    if (!ctx.dpy)
    {
        cerr << "XOpenDisplay failed\n";
        cerr << "Is X server running?\n";
        return EXIT_FAILURE;
    }

    // default values
    //hotkey next_window = {ShiftMask, XKeysymToKeycode(dpy, XStringToKeysym("Right")) };
    //hotkey prev_window = {ShiftMask, XKeysymToKeycode(dpy, XStringToKeysym("Left")) };    

    if (find_argument(argc, argv, "--next", s))
        ctx.next_window = parse_key(ctx.dpy, s);
    if (find_argument(argc, argv, "--prev", s))
        ctx.prev_window = parse_key(ctx.dpy, s);

    // get the X server root window
    Window root = XDefaultRootWindow(ctx.dpy);

    if (ctx.next_window.key)
    {
        int ret = XGrabKey(ctx.dpy, ctx.next_window.key, ctx.next_window.modifier, root, False, GrabModeAsync, GrabModeAsync);
        if (ret == BadAccess || ret == BadValue || ret == BadWindow)
            cerr << "Failed to register hotkey: next ";

        //XSync(ctx.dpy, False);
        //XFlush(ctx.dpy);
    }
    
    if (ctx.prev_window.key)
    {
        // for some reason the second key is not working. there's no error indicating the grabbing failed it just 
        // doesn't work. But this only happens when the application is started by upstart at system boot.
        // if started manually XGrabKey works as expected.
        // trying some silly hacking here to make this work...
        //XUngrabKey(ctx.dpy, ctx.prev_window.key, ctx.prev_window.modifier, root, False, GrabModeAsync, GrabModeAsync);
        //XSync(ctx.dpy, False);
        //XFlush(ctx.dpy);

        int ret = XGrabKey(ctx.dpy, ctx.prev_window.key, ctx.prev_window.modifier, root, False, GrabModeAsync, GrabModeAsync);
        if (ret == BadAccess || ret == BadValue || ret == BadWindow)
            cerr << "Failed to register hotkey: prev";
        
        //XSync(ctx.dpy, False);
        //XFlush(ctx.dpy);
    }

    XFlush(ctx.dpy);

    cerr.flush();

    // get a X11 file descriptor for event processing
    int fd = XConnectionNumber(ctx.dpy);

    XSetErrorHandler(x_error_handler);

    // for getting different events
    XSelectInput(ctx.dpy, root, SubstructureNotifyMask);
    XFlush(ctx.dpy);

    //bool reinstall_prev_key = true;

    // start event processing loop
    while (true)
    {
        fd_set read;
        FD_ZERO(&read);
        FD_SET(fd, &read);

        if  (select(fd + 1, &read, NULL, NULL, NULL) == -1)
        {
            if (errno != EINTR)
                cerr << "select failed: " << strerror(errno);
            break;
        }
        if (FD_ISSET(fd, &read))
            process_events(ctx);

        /*
        if (reinstall_prev_key && ctx.prev_window.key)
        {
            // desperate hacking
            // see the comments where the prev window key is originally registered...
            XUngrabKey(ctx.dpy, ctx.prev_window.key, ctx.prev_window.modifier, root, False, GrabModeAsync, GrabModeAsync);
            XSync(ctx.dpy, False);
            XFlush(ctx.dpy);
            
            int ret = XGrabKey(ctx.dpy, ctx.prev_window.key, ctx.prev_window.modifier, root, False, GrabModeAsync, GrabModeAsync);
            if (ret == BadAccess || ret == BadValue || ret == BadWindow)
                cerr << "Failed to register hotkey: prev";
        
            XSync(ctx.dpy, False);
            XFlush(ctx.dpy);

            reinstall_prev_key = false;
        }
        */
    }
    assert(ctx.dpy);
    XCloseDisplay(ctx.dpy);
    return EXIT_SUCCESS;
}
